source dsgetool.src
source varspectrum.src
source ssmspectrum.src
*******************************************************************************
*
* Replication program for Watson (1993), "Measures of Fit for Cailbrated Models",
* J. of Political Economy, vol 101, pp 1011-1041. This uses the model from
* King, Plosser and Rebelo, but with the date conventions used by Uhlig and
* Sims, rather than KPR's.
*
* FONC's (in balanced growth detrended variables) are
*
* theta/(1-N(t))=C(t)**(-eta)alpha*Y(t)/N(t)
* 1=(beta-star)*Et[(C(t)/C(t+1))**eta R(t+1)], where beta-star=beta*gamma**(1-eta)
* gamma R(t)=(1-alpha)*Y(t)/K(t-1)+1-delta
*
* with
*
* C(t)+I(t)=Y(t)
* gamma K(t)=I(t)+(1-delta)*K(t-1)
* Y(t)=Z(t)*K(t-1)**(1-alpha)*N(t)**alpha
* log Z(t)=(1-psi)*log(Z-bar)+psi*log(Z(t-1))+eps(t)
*
* State variables are k,i,y,c,n,r,z
*
* These will be the steady state solutions given the parameters
*
dec real k0 i0 y0 c0 n0 r0 z0
*
* These are the deep and auxiliary parameters
*
dec real theta beta eta delta alpha zbar rho gammax
*
*******************************************************************************
*
* The procedure SimsForm returns the rectangular arrays
* g0 and g1 in the Sims' canonical form g0 y(t) = g1 y(t-1) + ...
* You pass into it the steady state values around which the log
* linearization is to take place. This procedure needs to be
* rewritten for different models.
*
* gstate is the deviation of the original form of the equation from
* a steady state solution. These can be used to solve for the steady state
* numerically if there is no analytical solution.
*
procedure SimsForm xstate g0 g1 gstate
type vect xstate
type rect *g0 *g1
type vect *gstate
*
local real k0 i0 y0 c0 n0 r0 z0
local real temp
*
compute g0=%zeros(7,7)
compute g1=%zeros(7,7)
dim gstate(7)
*
compute k0=xstate(1),i0=xstate(2),y0=xstate(3),c0=xstate(4),$
  n0=xstate(5),r0=xstate(6),z0=xstate(7)
*
* Equation 1: y-c-i=0 identity
*
compute gstate(1)=y0-c0-i0
compute g0(1,2)=-i0
compute g0(1,3)=y0
compute g0(1,4)=-c0
*
* Equation 2: capital stock evolution
*
compute gstate(2)=gammax*k0-k0*(1-delta)-i0
compute g0(2,1)=gammax*k0
compute g0(2,2)=-i0
compute g1(2,1)=(1-delta)*k0
*
* Equation 3: production function
*
compute gstate(3)=y0-z0*k0**(1-alpha)*n0**alpha
compute g0(3,3)=1
compute g0(3,5)=-alpha
compute g0(3,7)=-1
compute g1(3,1)=(1-alpha)
*
* Equation 4: productivity shock
*
compute gstate(4)=(1-rho)*(log(z0)-log(zbar))
compute g0(4,7)=1
compute g1(4,7)=rho
*
* Equation 5: labor supply FONC
*
compute temp=c0**(-eta)*y0*(1-1.0/n0)
compute gstate(5)=theta/alpha-temp
compute g0(5,3)=temp
compute g0(5,4)=-eta*temp
compute g0(5,5)=temp/(n0-1)
*
* Equation 6: consumption Euler equation
*
compute temp=beta*gammax**(1-eta)*r0
compute gstate(6)=1-temp
compute g0(6,4)=-eta*temp
compute g0(6,6)=temp
compute g1(6,4)=-eta*temp
*
* Equation 7: interest rate
*
compute gstate(7)=gammax*r0-(1-alpha)*y0/k0-(1-delta)
compute g0(7,3)=-(1-alpha)*y0/k0
compute g0(7,6)=gammax*r0
compute g1(7,1)=-(1-alpha)*y0/k0
end
*
*******************************************************************************
*
* Using the g0, g1 and gstate arrays returned by SimsForm, this does a
* log-linearized Newton's method to solve for the steady state. You can fix
* some of the state variables in the solution by using the combination of
* row and col options. The row option extracts only a subset of the equations,
* while the col option extracts a subset of the variables. For instance,
*
* @SteadyState(cols=||1,2,3,4||,rows=||1,2,3,7||) xstate g0 g1
*
* solves equations 1,2,3 and 7 for variables 1,2,3 and 4, leaving variables
* 5,6 and 7 fixed at whatever values they have in the initial xstate vector.
*
procedure SteadyState xstate g0 g1
type vector *xstate
type rect *g0 *g1
*
option vect[int] row
option vect[int] col
*
local vect gstate dv
local integer iters i j m n
local vect[int] rowx colx
local rect a
local vect b
local real scale
*
compute n=%rows(g0)
*
if %defined(row).and.%defined(col)
   compute m=%rows(row),rowx=row,colx=col
else
   compute m=n,rowx=colx=%seq(1,n)

dim a(m,m) b(m)

*
* Solves for the steady state given the parameters
*
do iters=1,40
   compute scale=1.0-.8**iters
   @simsform xstate g0 g1 gstate
   ewise a(i,j)=g0(rowx(i),colx(j))-g1(rowx(i),colx(j))
   ewise b(i)=gstate(rowx(i))
   compute dv=-scale*%solve(a,b)
   if %normsqr(dv)<1.e-12
      return
   do i=1,m
      compute xstate(colx(i))=xstate(colx(i))*(1+dv(i))
   end do i
end do iters
end SteadyState

*
*******************************************************************************
*
* Parameters
*
* Calibration values used by Watson
*
compute eta=1.0      ;* Log utility
compute alpha=.58    ;* Labor's share
compute nbar=.2      ;* Steady state level of hours (used in place of "A" in utility function)
compute gammax=1.004 ;* Rate of technological progress
compute delta=.025   ;* Depreciation rate for capital (quarterly)
compute r=.065/4     ;* Quarterly steady state interest rate
compute rho=1.00     ;* AR coefficient in technical change
compute lrstd=.01    ;* Long-run standard deviation of output
compute zbar=1.0     ;* Mean of technology process
*
* Derived values
*
compute beta=gammax/(1+r)
compute sigma=lrstd*alpha
compute betastar=beta*gammax**(1-eta)
*
dec vect xstate(7)
compute k0=25,z0=zbar,n0=nbar,y0=z0*k0**(1-alpha)*n0**alpha,c0=.7*y0,i0=.3*y0,r0=1.0/beta
compute xstate(1)=k0,xstate(2)=i0,xstate(3)=y0,xstate(4)=c0,$
  xstate(5)=n0,xstate(6)=r0,xstate(7)=z0
*
* These are the pi and psi matrices used in Sims' form.
*
dec vect mpi(7) mpsi(7)
ewise mpi(i)=(i==6)
ewise mpsi(i)=(i==4)
*
* Solve for the steady state values pegging n, r and z, using equations 1, 2, 3 and 7
*
@SteadyState(cols=||1,2,3,4||,rows=||1,2,3,7||) xstate g0 g1
*
* Back out value of theta using nbar
*
compute theta=alpha*(1-nbar)/nbar*xstate(3)/xstate(4)
*
* Get data. Income, consumption and investment are log real per capita values
*
open data watson_jpe.rat
cal 1948 1 4
all 1988:4
data(format=rats) 1948:1 1988:4 y c invst h
*
* This is the number of frequency domain ordinates from 0 to pi.
*
compute nords = 64
*
* Use Sims method to get the DLM that represents the log linearized model
* State variables are k,i,y,c,n,r,z. Since we want y,c,i, and n, we pull
* those components out of the spectrum
*
@SimsSolve(t0=theta0,t1=theta1,cutoff=1.0/beta,pi=mpi,psi=mpsi) g0 g1
*
* Measures of fit calculations
*
* ACGF of the model
*
@SSMSpectrum(a=theta1,sw=%outerxx(theta0*sigma),ords=nords*2,components=||3,4,2,5||,factors=alphatilde) ssmspect
*
* Transform to the ACGF of the first differences
*
dec series[cmatrix] alphat_tran diff_trfunc
gset diff_trfunc 1 nords = %identity(4)*(1-%unit2(2-t,nords*2))
gset alphat_tran 2 nords = diff_trfunc*alphatilde
*
* ACGF of the data.
* Estimate a VAR with y-c and y-invst as cointegrating vectors
*
set w1 = y-y{1}
set w2 = y-c
set w3 = y-invst
set w4 = h
*
system(model=coint)
variables w1 w2 w3 w4
lags 1 to 6
det constant
end(system)
*
estimate(noprint) 1950:1 *
@VARSpectrum(factor=thetatilde,ords=nords*2) coint %sigma spectvar
*
* Transform into first differences in all variables
*
dec series[cmatrix] var_trfunc thetat_tran
*
gset var_trfunc 1 nords = ||1.0,0.0,0.0,0.0|1.0,-(1-%unit2(2-t,nords*2)),0.0,0.0|1.0,0.0,-(1-%unit2(2-t,nords*2)),0.0|0.0,0.0,0.0,1-%unit2(2-t,nords*2)||
gset thetat_tran 1 nords = var_trfunc*thetatilde
*
dec series[cmatrix] acgfxy acgfx acgfy acgfu
*
dec rect w
dec vect[rect] wlist(5)
compute wlist(1)=%identity(4)
compute wlist(2)=%outerxx(%unitv(4,1))
compute wlist(3)=%outerxx(%unitv(4,1))
compute wlist(4)=%outerxx(%unitv(4,1))
compute wlist(5)=%outerxx(%unitv(4,1))
dofor w = wlist
   *
   gset acgfxy 2 nords = (cxsvd=%cxsvd(%cxadj(thetat_tran)*w*alphat_tran)),alphat_tran*cxsvd(3)*%cxadj(cxsvd(1))*%cxadj(thetat_tran)
   gset acgfy  2 nords = thetat_tran*%cxadj(thetat_tran)
   gset acgfx  2 nords = alphat_tran*%cxadj(alphat_tran)
   gset acgfu  2 nords = acgfx+acgfy-acgfxy-%cxadj(acgfxy)
   *
   do i=1,4
      set spectxmodel 2 nords = %real(acgfx(t)(i,i))
      set spectxdata  2 nords = %real(acgfy(t)(i,i))
      set spectu      2 nords = %real(acgfu(t)(i,i))
      graph(nodates) 3
      # spectxmodel 2 nords
      # spectxdata  2 nords
      # spectu      2 nords
      sstats(mean) 2 nords spectu>>su spectxdata>>sy
      disp su/sy
   end do i
end dofor w

